/*
 * QO-100 250W PA Controller
 * (c) DJ0ABR / Kurt Moraw
 * License: GPL V3
 * 
 * Version 1.0
 * Date: June 2020
 * 
 * GPIO usage:
 * 
 * A0 ... measure 50V supply
 * A1 ... NTC from Ampleon Power Module
 * A2 ... reverse power
 * A3 ... forward power
 * A4 ... measure 5V supply
 * A5 ... measure 12V supply
 * A6 ... measure 24V supply
 * 
 * D2 ... PTT Input from TRX TX=low, RX=open
 * D3 ... PTT output to driver stage TX=low, RX=high
 * D4 ... FAN Relais control Off=high, On=low
 * D5 ... VG of power stage Off=low, On=High
 * 
 */

//#define DEBUG_OUTPUT
 
#define ADCREF      1070L    // mV

#define PTTIN       2
#define PTTDRIVER   3
#define FANPUMP     4
#define VG          5
#define MESS50      A0
#define MESSNTC     A1
#define MESSREV     A2
#define MESSPWR     A3
#define MESS5       A4 
#define MESS12      A5
#define MESS24      A6

long meas50 = 0;
long meas24 = 0;
long meas12 = 0;
long meas5 = 0;
long measntc = 0;
long measpwr = 0;
long measrev = 0;
int temperature = 0;
int pwr = 0;
int revpwr = 0;
int return_loss = 30;

int emergency = 0;
int transmitting = 0;

void setup() 
{
    init_gpio();
    init_adc();
    Serial.begin(2400);
    
    #ifdef DEBUG_OUTPUT
    Serial.println("");
    Serial.println("PA250 Controller RESET");
    #endif
    
    init_meanbuf();
}

void loop() 
{
    // measure 5,12,24 and 50 v supply
    meas_allvoltages();

    // calculate PA Temp in DegC
    temperature = cf_calc_temp(measntc);

    // calculate power and SWR
    coupler();

    // check for emergency condition
    check_emergency();

    // handle PTT
    handle_ptt();

    // handle Fan/Pump
    handle_fan();

    // print to serial
    #ifdef DEBUG_OUTPUT
        // print results as readable text
        debug_print();
    #else
        // send short binary message via serial interface
        send_binary();
    #endif
}

void init_gpio()
{
    // Fan/Pump output, default off=low
    pinMode(FANPUMP,OUTPUT);
    digitalWrite(FANPUMP,0);

    // VG output, default off=low
    pinMode(VG,OUTPUT);
    digitalWrite(VG,0);

    // Driver PTT output, default off=high
    pinMode(PTTDRIVER,OUTPUT);
    digitalWrite(PTTDRIVER,1);

    // PTT input
    pinMode(PTTIN,INPUT_PULLUP);
}

void init_adc()
{
    analogReference(INTERNAL);  // set internal 1.1v ref
}

long meas_adc(int port)
{
    // read 10 bit 0..1024
    long v = analogRead(port);

    // convert into voltage in mV
    return (v * ADCREF) / 1024L;
}

// Series/parallel resistors of the divider on the ADC inputs
#define R50S    (22000L + 22000L)
#define R50P    1000L

#define R24S    (22000L + 1000L - 100L) // -100L for adjustment
#define R24P    1000L

#define R12S    (12000L + 5600L + 150L) // +150L for adjustment
#define R12P    1200L

#define R5S    6800L
#define R5P    1200L

#define NUMVOLTAGES 7   // 7 voltages are measured
#define MEANBUFLEN  5   // mean value buffer length
long meanbuf[NUMVOLTAGES][MEANBUFLEN];

#define EMPTYBUFMARKER 999999L  // marker for reset condition

void init_meanbuf()
{
    for(int port=0; port<NUMVOLTAGES; port++)
    {
        for(int i=0; i<MEANBUFLEN; i++)
            meanbuf[port][i] = EMPTYBUFMARKER;
    }
}

void meas_mean(int vnum)
{
long val;

    // measure value;
    switch(vnum)
    {
        case 0: val = (meas_adc(MESS50) * (R50S + R50P)) / R50P; break;
        case 1: val = (meas_adc(MESS24) * (R24S + R24P)) / R24P; break;
        case 2: val = (meas_adc(MESS12) * (R12S + R12P)) / R12P; break;
        case 3: val = (meas_adc(MESS5) * (R5S + R5P)) / R5P; break;
        case 4: val = meas_adc(MESSNTC); break;
        case 5: val = meas_adc(MESSPWR); break;
        case 6: val = meas_adc(MESSREV); break;
    }

    // handle mean value buffer
    // shift all values to the end
    if(meanbuf[vnum][0] == EMPTYBUFMARKER)
    {
        // first value after reset
        for(int i=0; i<MEANBUFLEN; i++)
            meanbuf[vnum][i] = val;
    }
    else
    {
        // shift buffer
        for(int i=0; i<(MEANBUFLEN-1); i++)
            meanbuf[vnum][i+1] = meanbuf[vnum][i];
        // insert new value
        meanbuf[vnum][0] = val;
    }

    // calc mean value
    long vm = 0;
    for(int i=0; i<MEANBUFLEN; i++)
        vm += meanbuf[vnum][i];
    vm /= MEANBUFLEN;

    // save result
    switch(vnum)
    {
        case 0: meas50 = ((vm+50)/100)*100; break; // adjust to one decimal place
        case 1: meas24 = ((vm+50)/100)*100; break; 
        case 2: meas12 = ((vm+50)/100)*100; break; 
        case 3: meas5 = ((vm+50)/100)*100; break;
        case 4: measntc = vm; break;
        case 5: measpwr = vm; break;
        case 6: measrev = vm; break;
    }
}

void meas_allvoltages()
{
    for(int i=0; i<NUMVOLTAGES; i++)   
        meas_mean(i);
}

// calculate the temperature of the PA boad in deg.C
// v = voltage at the ADC input in mV
// returns: temp in deg.C.

#define RV      6800   // NTC to GND and RV to +VREF
#define VREF    5000

/*
NTC resistance:
R[ntc] = Umess * RV / (UV - Umess)
*/

// R table for NTC on Ampleon board ERTJ1VT152J
float temptab[] = {
    5949 , //    Ohms at 0 Grad usw...
    4430 , //    5
    3332 , //    10
    2530 , //    15
    1940 , //    20
    1500 , //    25
    1170  , //   30
    920  , //    35
    728  , //    40
    581  , //    45
    467  , //    50
    377  , //    55
    306  , //    60
    251  , //    65
    207  , //    70
    172  , //    75
    143  , //    80
    121  , //    85
    102   , //   90
    86   , //    95
    74   , //    100
    63   , //    105
    54   , //    110
    -1
};

// calculate the NTC temperature
// uin: voltage at ADC input in mV
// returns: deg.C

int cf_calc_temp(uint32_t uin)
{
float Rntc;
int i;
float x;
float Umess = (float)uin/1000.0;

    // Umess ist die Spannung am ADC Eingang in V
    // jetzt berechne daraus den Widerstand des NTCs
    Rntc = Umess * (float)RV / ((float)VREF/1000.0 - Umess);

    if(Rntc >= temptab[0]) return 0;   // lower than minimum value

    // suche den Bereich in der Tabelle
    i=0;
    while(temptab[i]!=-1)
    {
        if(temptab[i] <= Rntc) break;
        i++;
    }

    if(i==0) return 0; // kleiner als kleinster Wert

    if(temptab[i]!=-1)
    {
        // Widerstandsbereich gefunden, interpoliere
        x = i - (Rntc - temptab[i])/(temptab[i-1] - temptab[i]);

        // x ist jetzt der interpolierte Tabellenindex
        // rechne ihn in die Temperatur um
        return  (int)(x*5.0);
    }

    return 115; // größer als größter Wert
}

/*
 * ADPZ8313
 * 
 * 12.5mV per dB (inverse linear)
 * 
 * returns: power in dBm
 */

#define REFLEVEL    +33     // dBm
#define REFVOLTAGE  5450    // mV * 10
#define DELTA       125     // (mV*10) per dB

/*
 * returnloss to SWR formular, may take too much time
 * 
 * VSWR = (1+10^(-RL/20)) / (1-10^(-RL/20))
 */
 
int calc_power(int measmV)
{
    int udiff = measmV*10 - REFVOLTAGE;
    int dBdiff = udiff / DELTA;
    return REFLEVEL - dBdiff;
}

void coupler()
{
    // measure fwd and rev power in dBm
    pwr = calc_power(measpwr);
    revpwr = calc_power(measrev);

    // reflexion coefficient in dB
    return_loss = pwr - revpwr;

    if((pwr <=0 && revpwr <=0) || transmitting == 0)
    {
        pwr = 0;
        revpwr = 0;
        return_loss = 0;
    }
}


void handle_ptt()
{
int ptt;
static int oldptt = -1;

    // check PTT input
    ptt = digitalRead(PTTIN);

    // do only if PTT changed state
    if(ptt != oldptt)
    {
        if(ptt == 0 && emergency == 0)
        {
            // TXing
            digitalWrite(VG,1);
            delay(10);
            digitalWrite(PTTDRIVER,0);
            #ifdef DEBUG_OUTPUT
            Serial.println(">>> TRANSMIT <<<");
            #endif
        }
        else
        {
            // back to RX
            digitalWrite(PTTDRIVER,1);                        
            delay(10);
            digitalWrite(VG,0);

            #ifdef DEBUG_OUTPUT
            if(emergency == 0)
                Serial.println(">>> receive <<<");
            else
                Serial.println("Emergency SHUT OFF. Reset system to recover.");
            #endif
        }
        
        oldptt = ptt;
    }

    // just to be sure in case of problems:
    // set the static lines every time
    if(ptt == 0 && emergency == 0)
    {
        digitalWrite(VG,1);
        digitalWrite(PTTDRIVER,0);
        transmitting = 1;
    }
    else
    {
        digitalWrite(PTTDRIVER,1);                        
        digitalWrite(VG,0);

        transmitting = 0;
    }
}

int fan = 0;

void handle_fan()
{
static int oldfan = -1;

    if(temperature > 40 || transmitting)
    {
        // FAN on
        digitalWrite(FANPUMP,1);
        fan = 1; 
        if(oldfan == 0 || oldfan == -1)
        {
            #ifdef DEBUG_OUTPUT
            Serial.println(">>> FAN ON <<<");
            #endif
        }
    }

    if(temperature < 36 && !transmitting)
    {
        // FAN on
        digitalWrite(FANPUMP,0);
        fan = 0;
        if(oldfan == 1 || oldfan == -1)
        {
            #ifdef DEBUG_OUTPUT
            Serial.println(">>> FAN OFF <<<");
            #endif
        }
    }

    oldfan = fan;
}

/*
 * checks for emergency. Can ONLY be reset by a power-off
 */
void check_emergency()
{
    if(temperature > 60)
    {
        emergency = 1;
        #ifdef DEBUG_OUTPUT
        Serial.println("!!!!! Emergency: OVER TEMPERATURE");
        #endif
    }

    // check VSWR only during transmission with a power > 10 watts
    if(transmitting == 1 && pwr >= 40)
    {
        // a reflexion coeff of 10dB is about an VSWR of 2 : 1
        // if it less then 10dB then shut off
        if(return_loss < 10)
        {
            // shut PA OFF immediately
            digitalWrite(PTTDRIVER,1);                        
            delay(10);
            digitalWrite(VG,0);
            
            emergency = 1;
            #ifdef DEBUG_OUTPUT
            Serial.print("!!!!! Emergency: VSWR error: Pwr:");
            Serial.print(pwr);
            Serial.print(" Rev:");
            Serial.print(revpwr);
            Serial.print(" Loss:");
            Serial.println(return_loss);
            #endif
        }
    }
}

void debug_print()
{
static long old_meas50 = -1;
static long old_meas24 = -1;
static long old_meas12 = -1;
static long old_meas5 = -1;
static long old_pwr = -1;
static long old_revpwr = -1;
static int old_temperature = -1;

    if(old_meas50 == -1 || meas50 > (old_meas50+800) || meas50 < (old_meas50-800))
    {
        old_meas50 = meas50;
        Serial.print("50V: ");
        Serial.println((double)meas50 / 1000.0,1);
    }
    if(old_meas24 == -1 || meas24 > (old_meas24+500) || meas24 < (old_meas24-500))
    {
        old_meas24 = meas24;
        Serial.print("24V: ");
        Serial.println((double)meas24 / 1000.0,1);
    }
    if(old_meas12 == -1 || meas12 > (old_meas12+400) || meas12 < (old_meas12-400))
    {
        old_meas12 = meas12;
        Serial.print("12V: ");
        Serial.println((double)meas12 / 1000.0,1);
    }
    if(old_meas5 == -1 || meas5 > (old_meas5+200) || meas5 < (old_meas5-200))
    {
        old_meas5 = meas5;

        Serial.print("5V:  ");
        Serial.println((double)meas5 / 1000.0,1);
    }
    if(old_temperature == -1 || temperature > (old_temperature+3) || temperature < (old_temperature-3))
    {
        old_temperature = temperature;
        Serial.print("Temp:");
        Serial.println(temperature);
    }
    if(transmitting)
    {
        if(old_pwr == -1 || pwr > (old_pwr+2) || pwr < (old_pwr-2) || old_revpwr == -1 || revpwr > (old_revpwr+5) || revpwr < (old_revpwr-5) || return_loss < 10)
        {
            old_pwr = pwr;
            old_revpwr = revpwr;
            Serial.print("Pwr: ");
            Serial.print(measpwr);
            Serial.print(" = ");
            Serial.print(pwr);
            Serial.print("   Rev: ");
            Serial.print(measrev);
            Serial.print(" = ");
            Serial.print(revpwr);
            Serial.print("   RetL:");
            Serial.println(return_loss);
        }
    }
}

uint8_t chksum;

void send_int_binary_MSB_first(uint16_t v)
{
uint8_t m,l;

    m = (uint8_t)((v >> 8) & 0xff);
    l = (uint8_t)(v & 0xff);

    chksum += m;
    chksum += l;
    
    Serial.write(m);
    Serial.write(l);
}

void send_binary()
{
    chksum = 0;

    send_int_binary_MSB_first(0xffff);
    send_int_binary_MSB_first((uint16_t)meas50);
    send_int_binary_MSB_first((uint16_t)meas24);
    send_int_binary_MSB_first((uint16_t)meas12);
    send_int_binary_MSB_first((uint16_t)meas5);
    send_int_binary_MSB_first((uint16_t)temperature);
    send_int_binary_MSB_first((uint16_t)pwr);
    send_int_binary_MSB_first((uint16_t)revpwr);
    send_int_binary_MSB_first((uint16_t)transmitting);
    send_int_binary_MSB_first((uint16_t)emergency);

    Serial.write(chksum);
}
