/*  OCXO Controller
    (c) DJ0ABR
    GPL V.3

    compile with avr-gcc
    for ATmega88PA (ATmega88PB, ATmega8xxx)

    Fuses: X:FF H:D7 L:E2

    TxD is a debug output 9600n81
*/

#define BOARD4B_GPS // deactivate for board without GPS

#define F_CPU 8000000

#include <stdio.h> 
#include <stdarg.h> 
#include <stdlib.h> 
#include <string.h> 

#include <avr/io.h> 
#include <avr/interrupt.h>
#include <avr/eeprom.h>
#include <avr/pgmspace.h>
#include <util/delay.h> 
#include <util/twi.h>

void uart_printf(const char *fmt, ...);
uint64_t calcLast(uint32_t freqmid);
void setLED(char col);

enum _LEDCOL_ {
    red = 0,
    green,
    blue
};

char restart_counters = 0;
uint16_t meastime = 0;

// ====== system initialization ======

void gpio_init()
{
    // Port B initialization
    DDRB=(1<<DDB7) | (1<<DDB6) | (1<<DDB5) | (1<<DDB4) | (1<<DDB3) | (1<<DDB2) | (1<<DDB1) | (1<<DDB0);
    PORTB=(0<<PORTB7) | (1<<PORTB6) | (1<<PORTB5) | (1<<PORTB4) | (1<<PORTB3) | (1<<PORTB2) | (1<<PORTB1) | (1<<PORTB0);

    // Port C initialization
    DDRC=(0<<DDC6) | (1<<DDC5) | (0<<DDC4) | (0<<DDC3) | (0<<DDC2) | (0<<DDC1) | (0<<DDC0);
    PORTC=(0<<PORTC6) | (0<<PORTC5) | (0<<PORTC4) | (0<<PORTC3) | (0<<PORTC2) | (1<<PORTC1) | (1<<PORTC0);

    // Port D initialization
    DDRD=(0<<DDD7) | (0<<DDD6) | (0<<DDD5) | (0<<DDD4) | (0<<DDD3) | (0<<DDD2) | (1<<DDD1) | (0<<DDD0);
    PORTD=(1<<PORTD7) | (1<<PORTD6) | (1<<PORTD5) | (1<<PORTD4) | (1<<PORTD3) | (1<<PORTD2) | (0<<PORTD1) | (1<<PORTD0);
}

// values for 2ms (approx. value, the exact value is not required)
// 25MHz in 2ms gives 50000 pulses, which does not overflow the 16bit HW counters, so its ok
#define T1H 0xc4
#define T1L 0x32

void timer_init()
{
    // Timer/Counter 1 initialization
    // Clock source: System Clock / 8 = 1MHz
    // Mode: Normal count from TCNT1x to top=0xFFFF
    // Timer Period: 1 ms
    // Timer1 Overflow Interrupt: On
    TCCR1A=(0<<COM1A1) | (0<<COM1A0) | (0<<COM1B1) | (0<<COM1B0) | (0<<WGM11) | (0<<WGM10);
    TCCR1B=(0<<ICNC1) | (0<<ICES1) | (0<<WGM13) | (0<<WGM12) | (0<<CS12) | (0<<CS11) | (1<<CS10);
    TCNT1H=T1H;
    TCNT1L=T1L;
    ICR1H=0x00;
    ICR1L=0x00;
    OCR1AH=0x00;
    OCR1AL=0x00;
    OCR1BH=0x00;
    OCR1BL=0x00;
    TIMSK1=(0<<ICIE1) | (0<<OCIE1B) | (0<<OCIE1A) | (1<<TOIE1);
}

void TWI_init()
{
    TWCR = 4;   // enable TWI
    TWBR = 32;  // prescaler:32, makes TWI clock of 100k
    TWSR = 0;
}

// ====== DAC routines ======

#define DAC_addr 0xc0

uint8_t TWI_Status;

void Twi_wait_int()
{
    while((TWCR & 0x80) == 0)
    {
        //uart_printf("TWCR: %02X\n",TWCR);
    }
    //uart_printf("TWCR: %02X\n",TWCR);

    TWI_Status = TWSR & 0xf8;
    //uart_printf("TWSR: %02X\n",TWSR);
}

void TWI_sendbytes(uint8_t b2, uint8_t b3)
{
char error = 0;

    TWCR = 0xa4;    // Start Condition
    Twi_wait_int();

    if(TWI_Status == 0x08 || TWI_Status == 0x10)
    {
        TWDR = DAC_addr & 0xfe; // set slave address
        TWCR = 0x84;            // and send
        Twi_wait_int();

        if(TWI_Status == 0x18 || TWI_Status == 0x20)
        {
            TWDR = b2;              // set byte
            TWCR = 0x84;            // and send
            Twi_wait_int();

            if(TWI_Status == 0x28 || TWI_Status == 0x30)
            {
                // byte 2 sent
                TWDR = b3;              // set byte
                TWCR = 0x84;            // and send
                Twi_wait_int();

                if(TWI_Status == 0x28 || TWI_Status == 0x30)
                {
                    // byte 3 sent
                }
                else
                {
                    error = 3;
                    static const char txtP13[] PROGMEM  = {"error sending byte 3\n"};
                    uart_printf(txtP13);
                }
            }
            else
            {
                error = 2;
                static const char txtP12[] PROGMEM  = {"error sending byte 2\n"};
                uart_printf(txtP12);
            }

            TWCR = 0x94;    // Stop condition
        }

    }
    else
    {
        error = 1;
        static const char txtP11[] PROGMEM  = {"TWI bus occupied\n"};
        uart_printf(txtP11);
    }

    if(error != 0)
    {
        static const char txtP10[] PROGMEM  = {"TWI error: %d\n"};
        uart_printf(txtP10,error);
    }
}

void DAC_setvoltage(uint16_t dig)
{
uint8_t reg[2];

    if(dig >= 4096) dig = 4095;

    // set register values
    reg[0] = dig >> 8;
    reg[0] &= 0x0f;
    reg[1] = dig & 0xff;

    // send to device
    TWI_sendbytes(reg[0], reg[1]);
}

// ====== serial output routines ======

void uart_putchar(char c)
{
    loop_until_bit_is_set(UCSR0A, UDRE0);
    UDR0 = c;
}

void uart_printf(const char *fmt, ...)
{
static char s[120];

    char buf[strlen_P(fmt)+1];
    strcpy_P(buf, fmt);

	va_list ap;
	va_start(ap, fmt);
	vsprintf(s,buf,ap);
	va_end(ap);

    int len = strlen(s);
    if(len > sizeof(s)) sprintf(s,"str-err:%d\n",len);

	for(int i=0; i<strlen(s); i++)
      uart_putchar(s[i]);
}

void print_u64(uint64_t v64)
{
    uint32_t v64u = (uint32_t)(v64 / 100000);
    uint32_t v64l = (uint32_t)(v64 - (v64/100000)*100000);

    static const char txt[] PROGMEM  = "%lu%05lu";
    uart_printf(txt,v64u,v64l);
}

void uart_init(int baud)
{
    UBRR0L = (F_CPU / (16UL * baud)) - 1;
    UCSR0B = _BV(TXEN0);
}

// init string for UBLOX MAX-8Q to configure
// the 1PPS output for 10MHz
uint8_t gps_config_data[] = {
0xB5, 0x62, 0x06, 0x31, // Header UBX-CFG-TP5
0x20, 0x00,				// payload len = 32 Byte
0x00, 0x01, 0x00, 0x00, 0x32, 0x00, 0x00, 0x00,
//0x80, 0x96, 0x98, 0x00, // output freq if unlocked 10 MHz
0x00, 0x00, 0x00, 0x00, // output freq if unlocked OFF
0x80, 0x96, 0x98, 0x00,	// output freq if locked 10 MHz
//0x00, 0x00, 0x00, 0x80,	// pulse len ration if unlocked
0x00, 0x00, 0x00, 0x00,	// pulse len ration if unlocked
0x00, 0x00, 0x00, 0x80, // pulse len ration if locked
0x00, 0x00, 0x00, 0x00,	// unused user cfg
0xEF, 0x00, 0x00, 0x00, // flags
0, 0};			        // CHKSUM calculated before sending

void gps_init()
{
    // calc checksum
    uint8_t CK_A = 0, CK_B = 0;
	for(int i=2; i<sizeof(gps_config_data)-2; i++)
	{
		CK_A = CK_A + gps_config_data[i];
		CK_B = CK_B + CK_A;
	}
	gps_config_data[sizeof(gps_config_data)-2] = CK_A;
	gps_config_data[sizeof(gps_config_data)-1] = CK_B;

    // display UBLOX init string
    for(int i=0; i<sizeof(gps_config_data); i++)
    {
        static const char txtx[] PROGMEM  = "%02X ";
        uart_printf(txtx,gps_config_data[i]);
    }
    static const char txty[] PROGMEM  = "\n";
    uart_printf(txty);

    // send 20 times, just to be sure
    for(int rep=0; rep<20; rep++)
    {
        for(int i=0; i<sizeof(gps_config_data); i++)
            uart_putchar(gps_config_data[i]);

        _delay_ms(100);
    }
}

// ====== read contents of the four 74LV8154 registers ======

uint32_t getY()
{
    uint32_t v;

    #ifdef BOARD4B_GPS
    // data bit 0 on PC0, data bit 1 on PC1
        uint8_t v8d = PIND;
        v8d &= 0xfc;

        uint8_t v8c = PINC;
        v8c &= 3;

        v = v8d + v8c;



        //v = PIND & ~3;
        //v += (PINC & 3);
    #else
        // data bit 0 on PD0, data bit 1 on PC1
        v = PIND & ~2;
        v += (PINC & 2);
    #endif
    return v;
}

void gpio_read(uint16_t *pval, uint16_t *pref)
{
    // store counter values in register
    // rising edge of RCLK (PB7)
    PORTB = PORTB | 0x80;
    PORTB = PORTB & ~0x80;

    // read 25 MHz 
    // get Counter A (OCXO) high byte
    PORTB = PORTB & ~2;
    *pval = getY() << 8;
    PORTB = PORTB | 2;

    // get Counter A (OCXO) low byte
    PORTB = PORTB & ~1;
    *pval += getY();
    PORTB = PORTB | 1;

    // read 10 MHz Reference
    // get Counter A (OCXO) high byte
    PORTB = PORTB & ~0x40;
    *pref = getY() << 8;
    PORTB = PORTB | 0x40;

    // get Counter A (OCXO) low byte
    PORTB = PORTB & ~4;
    *pref += getY();
    PORTB = PORTB | 4;
}

// ====== Timer IRQ: read hardware counters every 1 ms and make counters for 10 and 25 MHz ======

uint64_t counter25;
uint64_t counter10;

#define MIDLEN  10
uint64_t midarr[MIDLEN];
char midpos = 0;

// 2ms Timer
ISR (TIMER1_OVF_vect)
{
static uint32_t diff25, diff10;
static uint16_t pval, pref;
static uint16_t poval, poref;
static char first=1;

    if(first || restart_counters)
    {
        // init counters
        first = 0;
        restart_counters = 0;
        counter25 = 0;
        counter10 = 0;
        gpio_read(&poval,&poref);
        for(char i=0; i<MIDLEN; i++)
            midarr[i] = 2500000000;
    }
    else
    {
        gpio_read(&pval,&pref);

        // calculate new counter, handle overflow
        if(pval == poval)
            diff25 = 0;
        else if(pval > poval)
            diff25 = (uint32_t)(pval - poval);
        else
            diff25 = ((uint32_t)pval + (uint32_t)0x10000 - (uint32_t)poval);

        if(pref == poref) 
            diff10 = 0;
        else if(pref > poref)
            diff10 = (uint32_t)(pref - poref);
        else
            diff10 = ((uint32_t)pref + (uint32_t)0x10000 - (uint32_t)poref);

        // increment counters
        counter25 += (uint64_t)diff25;
        counter10 += (uint64_t)diff10;

        poval = pval;
        poref = pref;
    }

    // make 1 second
    static int sec=0;
    if(++sec >= 500)
    {
        sec = 0;
        meastime++;
    }

    TCNT1H=T1H;
    TCNT1L=T1L;
}

// ====== calibrate the OCXO ======

char calstate = 0;          // 0=rapid search, 1=fast adapt, 2=slow adapt, 3=follow very slowly
int caltime = 10;           // calibration interval
int16_t dacvalue = 2048;    // start at mid level

char calnames[4][20] = {
    "rapid",
    "fast",
    "slow",
    "lock"
};

int16_t caltimes[4] = {
    30,
    100,
    140,
    200 // 700s meas time is maximum, longer times will overflow uint64_t at 25 MHz
};

int16_t multiplier[4] = {
    4,
    3,
    2,
    1
};

void eeprom_read_cal()
{
    uint16_t v1 = eeprom_read_word((uint16_t*)10);
    uint16_t v2 = eeprom_read_word((uint16_t*)20);
    uint16_t v3 = eeprom_read_word((uint16_t*)30);

    if(v1 == -1) v1=v2=v3=2048; // initial value if eeprom empty

    if(v1 == v2 && v2 == v3)
    {
        dacvalue = (int16_t)v1;
        if(dacvalue != -1) calstate = 1; //no rapid search if cal value available
    }

    if(dacvalue < 0) dacvalue = 0;
    if(dacvalue >= 4096) dacvalue = 4095;

    static const char txtP9[] PROGMEM  = {"\n\n\neeprom dacvalue: %d\n"};
    uart_printf(txtP9,v1);
}

void eeprom_write_cal()
{
    static const char txtP8[] PROGMEM  = {"eeprom write dacvalue: %d\n"};
    uart_printf(txtP8,dacvalue);
    eeprom_update_word((uint16_t*)0, (uint16_t)0);  // dummy
    eeprom_update_word((uint16_t*)10, (uint16_t)dacvalue);
    eeprom_update_word((uint16_t*)20, (uint16_t)dacvalue);
    eeprom_update_word((uint16_t*)30, (uint16_t)dacvalue);
}

void calibrate_ocxo(uint32_t freq)
{
    int16_t actdacval = dacvalue;

    int16_t diff = (uint16_t)((uint32_t)2500000000 - freq);
    int16_t adiff = abs(diff);

    // the maximum offset of the OCXO is specified in the data sheet as
    // +/-500ppb and +/-700ppb(Control Voltage) = +/-1300ppb
    // at 25MHz this is 32,5 Hz
    // so we ignore any value above 40 Hz offset as this may be a wrong measurement
    if(adiff >= 40)
    {
        static const char txtP7[] PROGMEM  = {"Frequency out of tuning range, offset: %d Hz\n"};
        uart_printf(txtP7,diff);
        return;
    }

    if(diff == 0 && calstate == 2)
    {
        eeprom_write_cal();
        setLED(green);
        calstate = 3;
    }
    else if(diff == 0 && calstate == 3)
    {
        // locked, nothing to do
    }
    else
    {
        dacvalue += diff * multiplier[calstate];
        DAC_setvoltage((uint16_t)dacvalue);

        if(calstate == 0 && adiff <= 50) 
        {
            calstate = 1;
        }
        else if(calstate == 1 && adiff <= 4) 
        {
            setLED(blue);
            calstate = 2;
        }
    }

    //uart_printf("Calib: act.freq: %05lu%05lu Hz, dacvalue was %d is %d, state: %s, next in %d s\n",freqH,freqL,actdacval,dacvalue,calnames[calstate],caltimes[calstate]);
    static const char txtP6[] PROGMEM  = {"%u %10lu %d %d %s %d\n"};
    uart_printf(txtP6,meastime,freq,actdacval,dacvalue,calnames[calstate],caltimes[calstate]);
}

/*
rt ... PB3
bl ... PB4
gn ... PB5
*/
void setLED(char col)
{
    switch (col)
    {
        case red: 
            PORTB &= ~8;
            PORTB |= 0x10;
            PORTB |= 0x20;
            break;
        case blue: 
            PORTB |= 8;
            PORTB &= ~0x10;
            PORTB |= 0x20;
            break;
        case green: 
            PORTB |= 8;
            PORTB |= 0x10;
            PORTB &= ~0x20;
            break;
    }
}

#define LASTKOMMALEN 10
uint8_t lastkomma[LASTKOMMALEN];

// #define TESTFREERUN // activate to measure OCXO during free run

int main()
{
    uint64_t c25, c10;
    char sf[30];
    char txt[50];
    char txtmid[50];

    // Crystal Oscillator division factor: 1
    CLKPR=(1<<CLKPCE);
    CLKPR=(0<<CLKPCE) | (0<<CLKPS3) | (0<<CLKPS2) | (0<<CLKPS1) | (0<<CLKPS0);

    gpio_init();
    TWI_init();
    uart_init(9600);

    static const char txtP1[] PROGMEM  = "\n\n\n\n\n\nOCXO Controller by DJ0ABR\n";
    uart_printf(txtP1);
    timer_init();
    gps_init();
    sei();

    // check calibration mode: PB3/PB4 connected
    DDRB=(1<<DDB7) | (1<<DDB6) | (1<<DDB5) | (1<<DDB4) | (0<<DDB3) | (1<<DDB2) | (1<<DDB1) | (1<<DDB0);
    PORTB &= ~0x10;
    _delay_ms(10);
    uint8_t pb3l = PINB & 8;
    PORTB |= 0x10;
    _delay_ms(10);
    uint8_t pb3h = PINB & 8;
    PORTB &= ~0x10;
    _delay_ms(10);
    uint8_t pb3l1 = PINB & 8;
    DDRB=(1<<DDB7) | (1<<DDB6) | (1<<DDB5) | (1<<DDB4) | (1<<DDB3) | (1<<DDB2) | (1<<DDB1) | (1<<DDB0);

    static const char txtPcal1[] PROGMEM  = {"%d %d %d\n"};
    uart_printf(txtPcal1,pb3l,pb3h,pb3l1);

    if(pb3l == 0 && pb3h == 8 && pb3l1 == 0)
    {
        // calibration mode, set DAC to mid value
        static const char txtPcal[] PROGMEM  = {"calibration mode. Set poti to 50.000000 MHz output\n"};
        uart_printf(txtPcal);
        DAC_setvoltage(2048);
    }
    else
    {
        // set initial value
        eeprom_read_cal();
        DAC_setvoltage((uint16_t)dacvalue);
    }

    //DAC_setvoltage(2900);
    //while(1);
/*
    // for test only: generate a saw tooth waveform
    while(1)
    {
        for(uint16_t i=0; i<4095; i++)
            DAC_setvoltage(i);
    }
*/  

    setLED(red);

    while (1)
    {
        // read actual counter values
        cli();
        c25 = counter25;
        c10 = counter10;
        sei();
/*
        // === TEST ===
        uint64_t freq64 = c25 * (uint64_t)1e9 / c10;

        midarr[midpos] = freq64;
        if(++midpos == MIDLEN) midpos=0;

        uint64_t freqmid64 = 0;
        for(char i=0; i<MIDLEN; i++)
            freqmid64 += midarr[i];
        freqmid64 /= MIDLEN;

        static const char txtt1[] PROGMEM  = {"tm: %d "};
        uart_printf(txtt1, meastime);
        print_u64(c25);
        static const char txtt2[] PROGMEM  = {" "};
        uart_printf(txtt2);
        print_u64(c10);
        uart_printf(txtt2);
        print_u64(freq64);
        uart_printf(txtt2);
        print_u64(freqmid64);
        static const char txtt3[] PROGMEM  = {"\n"};
        uart_printf(txtt3);

        _delay_ms(1000);


        continue;
        // ============
*/


        if(c10 < 1)
        {
            static const char txtP5[] PROGMEM  = {"free running, no ext. 10 MHz reference\n"};
            uart_printf(txtP5);
            _delay_ms(5000);
            restart_counters = 1;
            meastime = 0;
            continue;
        }

        // Calculate frequency for the actual measurement time
        uint32_t freq = (uint32_t)(c25 * (uint64_t)1e9 / c10);

        // frequency is in "freq"
        sprintf(sf,"%10lu",freq);
        sprintf(txt,"%c%c.%c%c%c%c%c%c,%c%c%c",sf[0],sf[1],sf[2],sf[3],sf[4],sf[5],sf[6],sf[7],sf[8],sf[9],sf[10]);

        // calc mean value if new freq value is within +/- 50Hz precision, otherwise ignore
        if(freq < 2500005000 && freq > 2499995000)
        {
            midarr[midpos] = freq;
            if(++midpos == MIDLEN) midpos=0;
        }
        uint64_t freqmid64 = 0;
        for(char i=0; i<MIDLEN; i++)
            freqmid64 += midarr[i];
        freqmid64 /= MIDLEN;
        uint32_t freqmid = (uint32_t)freqmid64;

        // average frequency is in "freqmid"
        sprintf(sf,"%10lu",freqmid);
        sprintf(txtmid,"%c%c.%c%c%c%c%c%c,%c%c%c",sf[0],sf[1],sf[2],sf[3],sf[4],sf[5],sf[6],sf[7],sf[8],sf[9],sf[10]);

        // calculate frequency error (in 100x ppb)
        int ferr = (int)((freqmid-2500000000)*40); // 40 = 1000/25;
        int fei = ferr / 100;
        int fef = abs(ferr - fei*100);

        static const char txtP2[] PROGMEM  = " Meas.Time:% 5d s  Frequency: %s Hz Mean-Frequency: %s Hz  Deviation: %d.%01d ppb          \n\033[1A";
        uart_printf(txtP2,caltimes[calstate]-meastime,txt,txtmid,fei,fef);
        
        #ifdef TESTFREERUN
        if(meastime >= 200)   // seconds
        #else
        if(meastime >= caltimes[calstate])   // seconds
        #endif
        {
            #ifdef TESTFREERUN
            static const char txtPfr[] PROGMEM  = {"%lu %05lu%05lu\n"};
            uart_printf(txtPfr,meastime,freqH,freqL);
            #else
            calibrate_ocxo(freqmid);
            #endif

            cli();
            restart_counters = 1;
            sei();
            meastime = 0;
        }

        _delay_ms(1000);    // value not important, just to slow down the display
        
    }
}
